/*  Server.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.dmud;

import com.mischiefbox.dmud.net.ConnectionHandler;
import com.mischiefbox.dmud.net.Listener;
import com.mischiefbox.dmud.util.InputQueue;
import com.mischiefbox.dmud.util.ProcessorThread;
import com.mischiefbox.dmud.util.Queue;

/**
 *  Implements a DMUD server.
 *
 *  @author Chris Jones
 *  @version $Id: Server.java,v 1.1.1.1 2001/06/27 01:33:17 cjones Exp $
 */
public class Server {
    /**
     *  The default port on which to listen.
     */
    public final static int SERVER_PORT = 8887;

    /**
     *  The default number of processor threads.
     *  <p>
     *  The number of processor threads is directly proportional to
     *  the number of simultaneous commands that may be executed.
     */
    public final static int PROCESSOR_THREAD_COUNT = 8;

    /**
     *  The default name of the processor thread group.
     *  <p>
     *  This is really only useful when debugging.
     */
    public final static String PROCESSOR_THREAD_GROUP = "Processor Threads";

    /**
     *  The server port on which to listen.
     */
    protected int iPort;

    /**
     *  The maximum number of processor threads.
     */
    protected int iProcessorThreadsMax;

    /**
     *  The name of the processor thread group.
     */
    protected String sProcessorThreadGroup;

    /**
     *  The common input queue.
     */
    protected InputQueue qInput;

    /**
     *  The common output queue.
     */
    protected Queue qOutput;

    /**
     *  The connection handler.
     */
    protected ConnectionHandler connectionHandler;

    /**
     *  The listener.
     */
    protected Listener listener;

    /**
     *  The processor thread group.
     */
    protected ThreadGroup tgProcessor;

    /**
     *  The array of processor threads.
     */
    protected ProcessorThread [] paProcessor;

    /**
     *  Create a new server.
     */
    public Server() {
        iPort = SERVER_PORT;
        iProcessorThreadsMax = PROCESSOR_THREAD_COUNT;
        sProcessorThreadGroup = PROCESSOR_THREAD_GROUP;
    }

    /**
     *  Set the server port.
     *  <p>
     *  This will not change the listening port on an already running
     *  server.
     *
     *  @param iPort the server port on which to listen.
     */
    public void setPort(int iPort) {
        this.iPort = iPort;
    }
    
    /**
     *  Set the maximum number of processor threads.
     *  <p>
     *  This will not change the number of threads in an already
     *  running server.
     *
     *  @param iThreads the number of processor threads to create.
     */
    public void setThreadCount(int iThreads) {
        iProcessorThreadsMax = iThreads;
    }

    /**
     *  Set the processor thread group name.
     *  <p>
     *  This will not change the processor thread group name in an
     *  already running server.
     *
     *  @param sGroup the thread group name.
     */
    public void setThreadGroupName(String sGroup) {
        this.sProcessorThreadGroup = sGroup;
    }

    /**
     *  Get the listener.
     *
     *  @return the listener if the server is running, null otherwise.
     */
    public Listener getListener() {
        return listener;
    }

    /**
     *  Start this server. Begin listening for connections and
     *  processing their request messages.
     */
    public void startServer() {
        // create the input queue
        qInput = new InputQueue();

        // create the output queue
        qOutput = new Queue();
    
        // create the processor thread group
        tgProcessor = new ThreadGroup(sProcessorThreadGroup);

        // create the array to hold references to the processor
        // threads
        paProcessor = new ProcessorThread[iProcessorThreadsMax];

        // create the processor threads
        for (int i = 0; i < iProcessorThreadsMax; i++) {
            paProcessor[i] = new ProcessorThread(tgProcessor, qInput, qOutput);
        }

        // create the connection handler
        connectionHandler = new ConnectionHandler(qInput, qOutput);

        // create the new listener
        listener = new Listener(connectionHandler, iPort);
    }

    /**
     *  Stop this server. Shut down the listener. Stop the connection
     *  handler. Kill the processor threads.
     */
    public void stopServer() {
        // stop listening
        if (listener != null && listener.isListening()) {
            listener.stop();
        }

        // stop the handler
        if (connectionHandler != null &&
            connectionHandler.isHandling()) {
            connectionHandler.stopHandler();
        }

        // shut down each processor thread
        for (int i = 0; i < paProcessor.length; i++) {
            paProcessor[i].shutdown();
        }

        // everything will now shut down gracefully on its own
    }

    /**
     *  Command line method to start the server.
     */
    public static void main(String [] args) {
        Server server = new Server();
        server.startServer();
    }
}
