/*  ConnectionHandler.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.dmud.net;

import com.mischiefbox.dmud.message.Message;
import com.mischiefbox.dmud.message.MessageTooLargeException;
import com.mischiefbox.dmud.util.InputQueue;
import com.mischiefbox.dmud.util.Queue;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 *  Responsible for keeping track of connection info objects, checking
 *  for new input on connections, and passing new output to
 *  connections.
 *  <p>
 *  Is similar to the functionality of the UNIX select() or poll() call.
 *  <p>
 *  Runs in a thread to allow network I/O to operate asynchronously
 *  from the rest of the system. Network I/O is checked for input
 *  first (on each connection), then attempts to empty the output
 *  queue.
 *
 *  @author Chris Jones
 *  @version $Id: ConnectionHandler.java,v 1.1.1.1 2001/06/27 01:33:17 cjones Exp $
 */
public class ConnectionHandler implements Runnable {
    /**
     *  The name of thread group for connection info objects.
     */
    public final static String CONNECTION_INFO_THREAD_GROUP =
      "Connection Info Group";

    /**
     *  The connection handler thread name.
     */
    public final static String CONNECTION_HANDLER_THREAD_NAME =
      "Connection Handler Thread";

    /**
     *  The sleep time for this thread (in between processing runs).
     */
    protected final static int SLEEP_MILLIS = 100;

    /**
     *  Registered connection info objects.
     */
    protected HashMap hmConnectionInfo;

    /**
     *  The input (incoming message) queue.
     */
    protected InputQueue qInput;

    /**
     *  The output (outgoing message) queue.
     */
    protected Queue qOutput;

    /**
     *  The handler thread.
     */
    protected Thread tHandler;

    /**
     *  Flag to indicate if this is currently handling I/O.
     *
     *  Set this to false to stop handling I/O and shutdown all the
     *  current connections.
     */
    protected boolean bHandling;

    /**
     *  The connection info object thread group.
     */
    protected ThreadGroup tgConnectionInfo;

    /**
     *  Create a new connection handler.
     *
     *  Do the housekeeping to ensure this is set up correctly.
     */
    public ConnectionHandler(InputQueue qInput, Queue qOutput) {
        this.qInput = qInput;
        this.qOutput = qOutput;
        
        hmConnectionInfo = new HashMap();

        // create the connection info thread group
        tgConnectionInfo = new
          ThreadGroup(CONNECTION_INFO_THREAD_GROUP);

        // start the network I/O processor
        tHandler = new Thread(this, CONNECTION_HANDLER_THREAD_NAME);
        tHandler.start();
    }

    /**
     *  Register a new connection info object.
     *  <p>
     *  This isn't synchronized because this should only be called
     *  from the new connection info object.
     */
    public void registerConnectionInfo(ConnectionInfo info) {
        hmConnectionInfo.put(info.getConnectionId(), info);
    }
    
    /**
     *  Unregister a connection info object.
     *  <p>
     *  This is synchronized (internally) so that it won't be 
     *  unregistered twice at once (unregister may be called from 
     *  more than one place) and so that the connection won't be used
     *  while it's being closed.
     */
    public void unregisterConnectionInfo(ConnectionInfo info) {
        // remove the connection info from the registered set
        synchronized (hmConnectionInfo) {
            hmConnectionInfo.remove(info.getConnectionId());
        }

        try {
            // flush the output buffer
            info.bos.flush();
        } catch (IOException e) {
            // ignore any errors as this is shutting down anyway
        }

        // terminate the connection
        info.terminate();
    }

    /**
     *  Indicates if this is handling I/O.
     *
     *  @return true if this is handling connection I/O.
     */
    public boolean isHandling() {
        return bHandling && tHandler != null;
    }

    /**
     *  Stops the handler from handling I/O.
     *  <p>
     *  Create a new handler to start handling I/O again. This will
     *  disconnect all the current clients.
     */
    public void stopHandler() {
        bHandling = false;
    }

    /**
     *  Process connection I/O, including checking for available
     *  messages from the client and from the message handler.
     */
    public void run() {
        bHandling = true;

        ConnectionInfo info = null;
        Message msg = null;

        while (bHandling) {
            // flush the output queue
            synchronized (qOutput) {
                if (qOutput.getSize() > 0) {
                    // get the next message
                    try {
                        msg = (Message)qOutput.dequeue();
                    } catch (NoSuchElementException e) {
                        // this should happen, but the queue was empty
                        System.err.println("ASSERT FAILED: Queue empty when it shouldn't be!");
                        break;
                    }
                }
            }
            
            if (msg != null) {
                // get the connection associated with the message
                info = 
                  (ConnectionInfo)hmConnectionInfo.get(msg.getConnectionId());
                    
                // tell the connection info to write the message
                info.writeMessage(msg.getBytes());
            }

            info = null;
            msg = null;
                    
            try {
                tHandler.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // ignore this
            }

            // TODO: walk the registered info objects and check their
            // idle times
        }
    }

    /**
     *  Pass a message on the behalf of the connection info to the
     *  input queue.
     *
     *  @param conInfo the connection info that received the message.
     *  @param baMsg the message received.
     *  @throws MessageTooLargeException when the input message
     *  exceeded protocol limits.
     */
    public void inputMessage(ConnectionInfo conInfo, byte [] baMsg)
    throws MessageTooLargeException {
        qInput.incomingMessage(conInfo.getConnectionId(), baMsg);
    }

    /** 
     *  Get the connection info thread group object.
     *
     *  @return the connection info thread group object.
     */
    public ThreadGroup getConnectionInfoThreadGroup() {
        return tgConnectionInfo;
    }
     
}

/*
 * $Log: ConnectionHandler.java,v $
 * Revision 1.1.1.1  2001/06/27 01:33:17  cjones
 * Initial load.
 *
 */
