/*  ConnectionInfo.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.dmud.net;

import com.mischiefbox.dmud.message.MessageTooLargeException;
import java.net.Socket;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;

/**
 *  Connection information object.
 *
 *  Provides information about a single connection. Handles the input
 *  for the connection in an asynchronous thread--passes the input to
 *  the connection handler which will delegate processing to the input
 *  queue object. Ouput is handled via an immediate method call from
 *  the connection handler.
 *
 *  @author Chris Jones
 *  @version $Id: ConnectionInfo.java,v 1.1.1.1 2001/06/27 01:33:17 cjones Exp $
 */
public class ConnectionInfo implements Runnable {
    /**
     *  The sleep time for this thread (in between checks for new
     *  input).
     */
    protected final static int SLEEP_MILLIS = 100;

    /**
     *  The maximum number of consecutive errors permitted for 
     *  the connection.
     */
    public final static int MAX_ERROR = 5;

    /**
     *  The connection handler that created and is responsible 
     *  for the connection.
     */
    protected ConnectionHandler handler;
    
    /**
     *  The Socket associated with the connection.
     */
    protected Socket sock;

    /**
     *  The input stream associated with the socket.
     */
    protected BufferedInputStream bis;

    /**
     *  The output stream associated with the socket.
     */
    protected BufferedOutputStream bos;

    /**
     *  The connection id (used to identify the connection).
     */
    protected String sConnectionId;

    /**
     *  The consecutive error count on the connection.
     *
     *  After MAX_ERROR errors, the connection will be dropped.
     */
    protected int iErrorCount;

    /**
     *  The last recorded time I/O was performed on the socket.
     *
     *  I/O may be performed without recording that it was done.
     */
    protected long lLastUseMillis;

    /**
     *  Indicates this is is processing input.
     */
    protected boolean bProcessing = false;

    /**
     *  The thread that processes input.
     */
    protected Thread tProcessor;

    /**
     *  Create a new connection info object.
     *
     *  The new ConnectionInfo object is responsible for registering
     *  itself with the ConnectionHandler if the info object
     *  initializes properly.
     *
     *  @param handler the connection handler that created the
     *                 connection info object.
     *  @param sock the socket this connection will use for I/O.
     *  @param iConnectionId the connection id.
     */
    public ConnectionInfo(ConnectionHandler handler, Socket sock, int
    iConnectionId) {
        this.handler = handler;
        this.sock = sock;
        this.sConnectionId = String.valueOf(iConnectionId);

        // attempt to set up the I/O streams
        // TODO: consider tuning the size of the buffered I/O streams
        try {
            bis = new BufferedInputStream(sock.getInputStream());
            bos = new BufferedOutputStream(sock.getOutputStream());

            // success, so register the connection info object
            handler.registerConnectionInfo(this);

            // start processing input
            tProcessor = new Thread(handler.getConnectionInfoThreadGroup(),
              this, sock.getInetAddress().getHostAddress());
            tProcessor.start();

            // set the last use time
            lLastUseMillis = System.currentTimeMillis();
        } catch (IOException e) {
            // unable to set up connection
            System.err.println("Unable to set up ConnectionInfo: " +
              e.getMessage());
        }
    }

    /**
     *  Process input on the socket.
     */
    public void run() {
        bProcessing = true;

        // the number of bytes available
        int iAvail;

        // the input message
        byte [] baMsg;
        
        while (bProcessing) {
            // default bytes available
            iAvail = 0;
        
            // get the number of bytes available
            try {
                iAvail = bis.available();
                resetError();
            } catch (IOException e) {
                // problem getting available bytes
                addError();
            }

            // allocate space for the input message
            baMsg = new byte[iAvail];

            // read the input bytes
            try {
                bis.read(baMsg, 0, iAvail);
                resetError();

                // pass the read message to the handler
                handler.inputMessage(this, baMsg);
            } catch (IOException e) {
                // problem getting bytes from connection
                addError();
            } catch (MessageTooLargeException e) {
                // message was too large
                System.err.println(e.getMessage());
                addError();
            }

            // sleep between reads
            try {
                tProcessor.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // ignore this
            }
        }
    
        // shut down the processor
        if (bis != null) {
            try {
                bis.close();
            } catch (IOException e) {
                // ignore this error
            } finally {
                bis = null;
            }
        }

        if (bos != null) {
            try {
                bos.close();
            } catch (IOException e) {
                // ignore this error
            } finally {
                bos = null;
            }
        }

        if (sock != null) {
            try {
                sock.close();
            } catch (IOException e) {
                // ignore this error
            } finally {
                sock = null;
            }
        }

        handler = null;
    }

    /**
     *  Accept a message to be written to the client socket.
     *
     *  @param baMessage the message to be written.
     *  @return true on successful write.
     */
    public boolean writeMessage(byte [] baMessage) {
        // the success indicator
        boolean bSuccess = false;

        try {
            // write the message to the output stream
            bos.write(baMessage, 0, baMessage.length);
            bos.flush();
            resetError();

            // indicate success
            bSuccess = true;
        } catch (IOException e) {
            // problem while writing bytes to the connection
            addError();
        }

        return bSuccess;
    }
     

    /**
     *  Indicates if this is processing input. Stop this by using the
     *  shutdown() method.
     *
     *  @return true if this is processing input.
     */
    public boolean isProcessing() {
        return bProcessing;
    }

    /**
     *  Get the connection id.
     *
     *  @return the connection id.
     */
    public String getConnectionId() {
        return sConnectionId;
    }

    /**
     *  Get the last used time (in milliseconds).
     *
     *  @return the last used time.
     */
    public long getLastUsedTime() {
        return lLastUseMillis;
    }

    /**
     *  Set the last used time to now.
     *
     *  Synchronized access to last used time field.
     */
    public synchronized void setLastUsedTime() {
        lLastUseMillis = System.currentTimeMillis();
    }

    /**
     *  Get the input stream.
     *
     *  @return the input stream for the connection.
     */
    public BufferedInputStream getInputStream() {
        return bis;
    }

    /**
     *  Get the output stream.
     *
     *  @return the output stream for the connection.
     */
    public BufferedOutputStream getOutputStream() {
        return bos;
    }

    /**
     *  Close the socket connection immediately.
     *
     *  This will not unregister the connection from the handler,
     *  nor will it do any housekeeping outside this class.
     *
     *  Intended to be called by the ConnectionHandler only! This
     *  should be the last call in the chain when a connection is shut
     *  down gracefully (the connection info object unregistered, etc).
     */
    public synchronized void terminate() {
        bProcessing = false;
    }

    /**
     *  Shut down the connection gracefully.
     *
     *  This will ask the handler to unregister this connection info.
     *  The handler will invoke the terminate() method and null the
     *  reference to the connection info object.
     */
    public void shutdown() {
        // if we have a valid handler reference, ask it to remove us
        // gracefully
        if (handler != null) {
            handler.unregisterConnectionInfo(this);
        } else {
            // we should never have this situation occur
            // no valid handler, so terminate
            System.err.println("ConnectionInfo was missing a valid handler, so terminating without unregistering.");
            terminate();
        }
    }

    /**
     *  Get the number of errors on the connection.
     *
     *  @return the number of consecutive errors on this connection.
     */
    public int getErrors() {
        return iErrorCount;
    }

    /**
     *  Add to and check the error count.
     */
    public void addError() {
        iErrorCount++;

        if (iErrorCount > MAX_ERROR) {
            // shut this connection down due to too many errors
            System.err.println("Shutting down connection " + 
              sConnectionId + " because of too many errors.");
            shutdown();
        }
    }

    /**
     *  Reset the error count to zero (success in I/O).
     */
    public void resetError() {
        iErrorCount = 0;
    }
    
    /**
     *  Finalizer, in case this goes out of scope.
     */
    protected void finalize() throws Throwable {
        // free up the network resources
        shutdown();
    }
}

/*
 * $Log: ConnectionInfo.java,v $
 * Revision 1.1.1.1  2001/06/27 01:33:17  cjones
 * Initial load.
 *
 */
