/*  InputQueue.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.dmud.util;

import com.mischiefbox.dmud.message.Message;
import com.mischiefbox.dmud.message.MessageTooLargeException;
import java.util.Hashtable;

/**
 *  Input queue and manager.
 *
 *  Accepts partial or complete messages. Holds partial messages until
 *  a complete message is composed, or discards messages beyond a
 *  hard-coded message size limit. Complete messages become a new
 *  Message object which are put onto the message handler queue.
 *
 *  This class is a specialization of the generic Queue class.
 *
 *  The queue is a linked list. New elements are added to the tail,
 *  available elements are removed from the front.
 *
 *  @see Message
 *  @see Queue
 *
 *  @author Chris Jones
 *  @version $Id: InputQueue.java,v 1.1.1.1 2001/06/27 01:33:17 cjones Exp $
 */
public class InputQueue extends Queue {
    /**
     *  The maximum incomplete message size.
     */
    protected static final int MAX_MESSAGE_SIZE = 65536;
    
    /**
     *  The dictionary (Hashtable) of partial messages.
     *
     *  A Hashtable is used because the class is thread-safe
     *  (synchronized so that multiple connection handlers can work
     *  with it at once).
     *
     *  Key is connection id, value is a byte array partial message.
     */
    protected Hashtable htMessages;
    
    /**
     *  Message header byte signature.
     *
     *  Used in partial-message matching.
     */
    protected static byte [] baMessageHeader;

    /**
     *  Message footer byte signature.
     *
     *  Used in partial-message matching.
     */
    protected static byte [] baMessageFooter;
    
    /**
     *  Create a new InputQueue.
     *
     *  Do the housekeeping necessary to make the input queue usable.
     */
    public InputQueue() {
        super();
        // initialize the hashtable with the default size
        htMessages = new Hashtable();
    }
    
    /**
     *  Convert partial messages into Message objects.
     *
     *  Accepts a partial message and connection. Attempts to
     *  create a complete Message object from the partial message and
     *  previous partial messages. Inserts the completed message object 
     *  onto the queue. Discards incomplete messages that are larger
     *  than the maximum message size.
     *
     *  Partial messages are stored in a dictionary-type object
     *  (Hashtable, so multiple connection handlers can use this
     *  method).
     *
     *  @param sConnectionId the incoming connection id.
     *  @param baMessageContent the incoming message content.
     *  @throws MessageTooLargeException on overly long messages.
     */
    public void incomingMessage(String sConnectionId, byte []
    baMessageContent) throws MessageTooLargeException {
        // index of the beginning of the complete message
        int iMessageBegin;

        // index of the end of the complete message
        int iMessageEnd;

        // size of the partial message remaining after processing a
        // message
        int iPartialRemaining;

        // partial message that will be used for processing
        byte [] baMessagePartial;

        // temporary byte array for message copying/resizing
        byte [] baMessageTmp;
    
        // indicates that there is more of the partial message to scan
        boolean bMore;

        // look for a partial message in the hashtable
        baMessagePartial = (byte [])htMessages.get(sConnectionId);

        if (baMessagePartial == null) {
            // size the baMessagePartial to hold the incoming message
            // copy the incoming message into the partial array
            baMessagePartial = new byte[baMessageContent.length];
            System.arraycopy(baMessageContent, 0, baMessagePartial, 0,
              baMessageContent.length);
        } else {
            // resize the partial message array to contain the partial
            // message plus the incoming message array and concatenate
            // the byte arrays
            baMessageTmp = new byte[baMessagePartial.length];
            System.arraycopy(baMessagePartial, 0, baMessageTmp, 0,
              baMessagePartial.length);
            baMessagePartial = new byte[baMessagePartial.length +
              baMessageContent.length];
            System.arraycopy(baMessageTmp, 0, baMessagePartial, 0, 
              baMessageTmp.length);
            System.arraycopy(baMessageContent, 0, baMessagePartial,
              baMessageTmp.length, baMessageContent.length);
            baMessageTmp = null;
        }
        
        // Scan the partial message byte array for a complete message,
        // create a new message object and enqueue it if found; check
        // the partial message for problems (like footers without
        // headers or a too-large message buffer) and correct them.
        do {
            // by default, don't scan again
            bMore = false;
            
            // scan the incoming message for a complete message
            // look for a valid header
            iMessageBegin = scanByteArray(baMessagePartial,
              baMessageHeader);
            // look for a valid footer
            iMessageEnd = scanByteArray(baMessagePartial,
              baMessageFooter);
            if (iMessageEnd > -1) {
                // reset message end to the actual end of the message
                iMessageEnd += baMessageFooter.length;
            }

            // set the size of the remining message, assuming there is
            // a complete message (or there is some other condition
            // that will cause the partial message to be modified)
            iPartialRemaining = baMessagePartial.length - iMessageEnd;
            
            // is there a complete message inside the partial?
            if (iMessageBegin > -1 && iMessageEnd > -1 && 
                iMessageEnd > iMessageBegin) {
                // yes, extract the valid message
                // pass the valid message into a new Message object
                baMessageTmp = new byte[iMessageEnd - iMessageBegin];
                System.arraycopy(baMessagePartial, iMessageBegin, baMessageTmp,
                  0, baMessageTmp.length);

                Message msg = new Message(sConnectionId, new
                  String(baMessageTmp));

                // enqueue the Message object so it may be processed
                enqueue(msg);
                
                // rewrite the partial message so it starts at the end of the
                // complete message
                if (iPartialRemaining > 0) {
                    baMessageTmp = new byte[baMessagePartial.length -
                      iMessageEnd];
                    System.arraycopy(baMessagePartial, iMessageEnd,
                      baMessageTmp, 0, baMessageTmp.length);
                    baMessagePartial = new byte[baMessageTmp.length];
                    System.arraycopy(baMessageTmp, 0,
                      baMessagePartial, 0, baMessagePartial.length);
                      
                    // indicate that there is more message to scan
                    bMore = true;
                } else {
                    // done processing, set the baMessagePartial to
                    // null
                    baMessagePartial = null;
                }
            } else {
                if (iMessageBegin == -1 && iMessageEnd > -1) {
                    // a footer is missing a header
                    // if there is content to the partial message?
                    if (iPartialRemaining > 0) {
                        // truncate the partial to the content following
                        // the end of the footer
                        baMessageTmp = new
                          byte[baMessagePartial.length - iMessageEnd];
                        System.arraycopy(baMessagePartial, iMessageEnd, 
                          baMessageTmp, 0, baMessageTmp.length);
                        baMessagePartial = new
                          byte[baMessageTmp.length];
                        System.arraycopy(baMessageTmp, 0,
                          baMessagePartial, 0, baMessagePartial.length);

                        // indicate that we should scan again
                        bMore = true;
                    } else {
                        // the end of the footer was the end of the
                        // message so null the partial message
                        baMessagePartial = null;
                    }
                } else if (iMessageBegin > iMessageEnd) {
                    // a footer may preceed a header or a footer may
                    // be missing
                    if (iMessageEnd != -1) {
                        // a footer that preceeds a header
                        // truncate the partial to the content
                        // following the end of the footer
                        baMessageTmp = new
                          byte[baMessagePartial.length - iMessageEnd];
                        System.arraycopy(baMessagePartial, iMessageEnd, 
                          baMessageTmp, 0, baMessageTmp.length);
                        baMessagePartial = new
                          byte[baMessageTmp.length];
                        System.arraycopy(baMessageTmp, 0,
                          baMessagePartial, 0, baMessagePartial.length);

                        // indicate that we should scan again
                        bMore = true;
                    }
                    // else, the message is incomplete and the footer
                    // is missing
                }
                // else, there was an incomplete message that we can't
                // completely process yet, so don't scan again (bMore
                // is already false)
            }
        } while (bMore);

        // check the partial message to see if it exceeds the legal
        // message size
        if (baMessagePartial != null && baMessagePartial.length >
            MAX_MESSAGE_SIZE) {
            // discard the partial message from the queue
            htMessages.put(sConnectionId, baMessagePartial);
            
            // throw a new message too large exception
            throw new MessageTooLargeException(sConnectionId);
        }
        
        if (baMessagePartial != null) {
            // store the remaining partial message in the dictionary
            htMessages.put(sConnectionId, baMessagePartial);
        } else {
            // remove the message
            htMessages.remove(sConnectionId);
        }
    }

    /**
     *  Scan a byte array for a pattern.
     *
     *  Arrays in Java may not be larger than int (32-bit), so it is
     *  safe to return int from this method.
     *
     *  @param baContent the searched byte array.
     *  @param baSpec the specified byte array to find.
     *  @return the offset of the specified byte array inside the
     *          searched byte array, or -1 if not found.
     */
    public static int scanByteArray(byte [] baContent, byte [] baSpec) {
        // make sure that the search spec and content are both
        // non-empty
        if (baContent.length == 0 && baSpec.length == 0 &&
            baContent.length < baSpec.length) {
            // baContent cannot contain baSpec
            return -1;
        }
         
        // determine the last byte index in the content that may be
        // searched
        int iLastIndex = baContent.length - baSpec.length;
        
        // the spec index
        int iSpec;

        // search the byte array
        for (int i = 0; i <= iLastIndex; i++) {
            // does the byte under consideration match the spec
            // string?
            for (iSpec = 0; iSpec < baSpec.length; iSpec++) {
                if (baContent[i + iSpec] != baSpec[iSpec]) {
                    // break out of inner for-loop and get the next
                    // character
                    break;
                }

                // are we at the end of the spec string?
                if (iSpec == baSpec.length - 1) {
                    // we found the pattern
                    return i;
                }
            }
        }

        // pattern wasn't found
        return -1;
    }

    /**
     *  Static initializer.
     *
     *  Prepare the incoming message hashtable.
     */
    static {
        // fill in the message header and footer byte signatures
        baMessageHeader = Message.HEADER.getBytes();
        baMessageFooter = Message.FOOTER.getBytes();
    }

}

/*
 * $Log: InputQueue.java,v $
 * Revision 1.1.1.1  2001/06/27 01:33:17  cjones
 * Initial load.
 *
 */
