/*  ProcessorThread.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.dmud.util;

import com.mischiefbox.dmud.message.Message;
import com.mischiefbox.dmud.util.Queue;
import java.util.NoSuchElementException;
import org.python.util.PythonInterpreter;

/**
 *  Defines a message processor thread.
 *
 *  @author Chris Jones
 *  @version $Id: ProcessorThread.java,v 1.1.1.1 2001/06/27 01:33:17 cjones Exp $
 */
public class ProcessorThread implements Runnable {
    /**
     *  Idle thread sleep time (between waking up to check status).
     */
    protected int SLEEP_MILLIS = 75;

    /**
     *  The processor thread.
     */
    protected Thread tProcessor;

    /**
     *  The message to process.
     */
    protected Message msg;

    /**
     *  The last used time of this processor thread.
     */
    protected long lLastUsedMillis;

    /**
     *  The time when this will be idle too long and be shut down
     *  automatically.
     */
    protected long lShutdownMillis;

    /**
     *  Indicates if this is currently in use (active).
     */
    protected boolean bActive;

    /**
     *  Indicates if this is processing (thread started).
     */
    protected boolean bProcessing;

    /**
     *  The processor thread's python interpreter.
     */
    protected PythonInterpreter pyInterp;

    /**
     *  The input queue with new messages to be processed.
     */
    protected Queue qInput;

    /**
     *  The output queue that will receive messages processed by this
     *  thread.
     */
    protected Queue qOutput;

    /**
     *  The thread group that contains this processor thread.
     */
    protected ThreadGroup tgManager;

    /**
     *  Create a new processor thread.
     *
     *  @param tgManager the thread group that owns this thread.
     *  @param qInput the input queue on which new messages are found.
     *  @param qOutput the output queue which received processed
     *         messages.
     */
    public ProcessorThread(ThreadGroup tgManager, Queue qInput, Queue qOutput) {
        this.tgManager = tgManager;
        this.qInput = qInput;
        this.qOutput = qOutput;
        
        // create the python interpreter
        pyInterp = new PythonInterpreter();
        
        // TODO: consider initialization to set the interpreter up to
        // a known state (Python)
        
        tProcessor = new Thread(tgManager, this);
        tProcessor.start();

        // set the last used time
        lLastUsedMillis = System.currentTimeMillis();
        
        // set the active flag
        bActive = false;
    }

    /**
     *  Process the message and put the results (if any) on the output
     *  queue.
     */
    public void run() {
        // mark this as processing
        bProcessing = true;
        
        // the message to be processed
        Message msg = null;
        
        while (bProcessing) {
            // look for an available message on the input queue
            try {
                msg = (Message)qInput.dequeue();

                // mark this as doing work
                bActive = true;
            } catch (NoSuchElementException e) {
                // no message available...ignore this exception
            }
        
            if (msg != null) {
                // TODO: flesh this out

                // determine if the message is a system message or
                // invoked on an object in the interpreter
                // if (msg instanceof SystemMessage) {
                //     msg.doMessage();
                // } else {
                //     qOutput.enqueue(invokePythonMessage(msg));
                // }
                
                // TODO: remove this temporary code, but remember to
                // enqueue the return message
                qOutput.enqueue(msg);


                // clear the processor variables
                msg = null;

                // set the last used time
                lLastUsedMillis = System.currentTimeMillis();
    
                // set this as inactive (not doing work)
                bActive = false;
            } 

            // sleep for a period of time
            try {
                tProcessor.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // woken up, so we should have something to do
            }
        } // end while

        // release the Python interpreter
        pyInterp = null;
    }

    /**
     *  Shut down the processor thread.
     */
    public void shutdown() {
        // stop this from processing
        bProcessing = false;

        // dereference the thread so it can be GCed
        tProcessor = null;
    }

    /**
     *  Indicates if this is actively processing.
     *
     *  @return true if this is active, false otherwise.
     */
    public boolean isActive() {
        return bActive;
    }

    /**
     *  Gets the last used time.
     *
     *  @return the last used time (in milliseconds).
     */
    public long getLastUsed() {
        return lLastUsedMillis;
    }
}
