/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

import java.util.*;

public class Attlist {
  /**
   *  The associated element name (from the DTD).
   */
  private String sElementName;
  
  /**
   *  The associated element (set by the DTD object).
   */
  private Element element;
  
  /**
   *  The attributes from the DTD.
   */
  private String sAttlist;
  
  /**
   *  The parsed attribute list.
   */
  private Vector vAttlist;

  /**
   *  Constants for parsing the DTD string.
   */
  private final static String DTD_CDATA = "CDATA";
  private final static String DTD_ID = "ID";
  private final static String DTD_IDREF = "IDREF";
  private final static String DTD_NOTATION = "NOTATION";
  private final static String DTD_ENTITY = "ENTITY";
  private final static String DTD_ENTITIES = "ENTITIES";
  private final static String DTD_NMTOKEN = "NMTOKEN";
  private final static String DTD_NMTOKENS = "NMTOKENS";
  private final static String DTD_IMPLIED = "#IMPLIED";
  private final static String DTD_REQUIRED = "#REQUIRED";
  private final static String DTD_FIXED = "#FIXED";

  /**
   *  Creates a new Attlist (attribute list) object.
   *
   *  @param elementName The element name from the DTD.
   *  @param attlist The attribute list from the DTD.
   */
  public Attlist(String elementName, String attlist) {
    this.sElementName = elementName;
    this.sAttlist = attlist;
    vAttlist = new Vector();

    parseAttribute(this.sAttlist);

    // debugging only
    //    System.out.println(toString());
  }

  /**
   *  Get the element.
   *
   *  @return The element associated with this attribute list.
   */
  public Element getElement() {
    return element;
  }

  /**
   *  Set the element for this attribute.
   *
   *  @param element The element for this attribute.
   */
  public void setElement(Element element) {
    this.element = element;
  }

  /**
   *  Get the element name from the DTD.
   *
   *  @return The element name.
   */
  String getElementName() {
    return sElementName;
  }

  /**
   *  Get the attribute list.
   *
   *  @return The attribute list.
   */
  public Vector getAttributes() {
    return vAttlist;
  }

  /**
   *  Merge another attribute list into this list.
   *
   *  @param attlist The other attribute list.
   */
  public void merge(Attlist attlist) {
    Vector vAttl = attlist.getAttributes();
    Attribute attr = null;
    boolean bInsert;

    // iterate across the new attributes to add
    for (int i = 0; i < vAttl.size(); i++) {
      attr = (Attribute)vAttl.elementAt(i);
      bInsert = true;

      // make sure that the element is not already present
      for (int j = 0; j < this.vAttlist.size(); j++) {
	if (attr.equals((Attribute)this.vAttlist.elementAt(j))) {
	  bInsert = false;
	}
      }

      if (bInsert) {
	this.vAttlist.addElement(attr);
      }
    }
  }

  /**
   *  Parse the attribute list. The rules are fun and typically
   *  complex.
   *
   *  @param attributes The attribute list from the DTD (unparsed).
   */
  private void parseAttribute(String attributes) {
    // General rules for parsing attributes:
    // The tag looks like:
    //    !ATTLIST elem param type mode value
    //
    // 'elem param type mode value' can repeat.
    // '!ATTLIST' has been pulled already.
    // 'elem' is required and will only show up ONCE. 
    //    It has been pulled already.
    // 'param' must be unique and indicates that start of a new attribute
    // 'type' is required and is one of:
    //    CDATA    (character data)
    //    (x | y)  (user defined)
    //    ID       (gives the ID of this element)
    //    IDREF    (points to the ID of the speced element)
    //    NOTATION (x | y)  (enumeration of notation types)
    //    ENTITY   (single unparsed entity)
    //    ENTITIES (multiple unparsed entities)
    //    NMTOKEN  (single token value)
    //    NMTOKENS (multiple token values)
    // 'mode' is optional and can be one of
    //    #IMPLIED (default)
    //    #REQUIRED
    //    #FIXED
    // 'value' is optional in cases except where mode is #FIXED
    //    It provides a default value for the attribute.
    
    String sParam = "";
    int iType = Attribute.USER_DEFINED;
    int iMode = Attribute.IMPLIED;
    String sValue = "";
    Vector vEnum = new Vector();    // for enumerated types

    Attribute attr = null;

    // the index of the pointer into the attributes String
    int i = 0;
    
    // temporary index
    int j = 0;

    // word buffer
    String sWord = "";

    // indicates that the current parameter is ready
    boolean done = false;
    boolean touched = false;

    while (i < attributes.length()) {
      // get the data type
      if (attributes.charAt(i) == '(') {
	// this is a enumerated type
	// get the end of the tag (closing parenthesis)
	j = attributes.indexOf(")", i);
	vEnum = makeUserTypeEnum(attributes.substring(i + 1, j));
	i = j;
      } else {
	// build the word
	if (attributes.charAt(i) == ' ') {
	  // end of word!
	  // clean up and leading/trailing spaces
	  sWord = sWord.trim();
	  
	  // ensure that we're not comparing an empty string
	  if (sWord.length() > 0) {
	    // does the word match?
	    if (sWord.equals(DTD_CDATA)) {
	      iType = Attribute.CDATA;
	      touched = true;
	    } else if (sWord.equals(DTD_ID)) {
	      iType = Attribute.ID;
	      touched = true;
	    } else if (sWord.equals(DTD_IDREF)) {
	      iType = Attribute.IDREF;
	      touched = true;
	    } else if (sWord.equals(DTD_NOTATION)) {
	      iType = Attribute.NOTATION;
	      touched = true;
	    } else if (sWord.equals(DTD_ENTITY)) {
	      iType = Attribute.ENTITY;
	      touched = true;
	    } else if (sWord.equals(DTD_ENTITIES)) {
	      iType = Attribute.ENTITIES;
	      touched = true;
	    } else if (sWord.equals(DTD_NMTOKEN)) {
	      iType = Attribute.NMTOKEN;
	      touched = true;
	    } else if (sWord.equals(DTD_NMTOKENS)) {
	      iType = Attribute.NMTOKENS;
	      touched = true;
	    } else if (sWord.equals(DTD_IMPLIED)) {
	      iMode = Attribute.IMPLIED;
	      touched = true;
	    } else if (sWord.equals(DTD_REQUIRED)) {
	      iMode = Attribute.REQUIRED;
	      touched = true;
	    } else if (sWord.equals(DTD_FIXED)) {
	      iMode = Attribute.FIXED;
	      touched = true;
	    } else {
	      if (sWord.startsWith("\"")) {
		// got a default value
		sValue = sWord.substring(1, sWord.length() - 1);
		done = true;
	      } else {
		// we got the parameter for the following element
		// back up the iterator
		if (!done && touched) {
		  done = true;
		  // back up the iterator
		  i = i - sWord.length() - 1;
		  sWord = "";
		} else {
		  sParam = new String(sWord);
		  touched = false;
		}
	      }
	      
	      // if we're done
	      if (done) {
		// make the new attribute
		attr = new Attribute(sParam, iType, iMode, sValue, vEnum);
		vAttlist.addElement(attr);
		
		// reset the values
		sParam = "";
		iType = Attribute.USER_DEFINED;
		iMode = Attribute.IMPLIED;
		sValue = "";
		vEnum = new Vector();
		done = false;
		touched = false;
	      } // end if done
	    }

	    // reset the sWord so words don't get concatenated
	    sWord = "";
	  } // end if word length > 0
	  
	} else {
	  sWord += attributes.substring(i, i + 1);
	}
      }

      // increment character under consideration
      i++;
    } // end while i < length

    if (touched) {
      // make the new attribute
      attr = new Attribute(sParam, iType, iMode, sValue, vEnum);
      vAttlist.addElement(attr);
      
      // reset the values
      sParam = "";
      iType = Attribute.USER_DEFINED;
      iMode = Attribute.IMPLIED;
      sValue = "";
      vEnum = new Vector();
      done = false;
    }
  }

  /**
   *  Turn the user defined type from the DTD into a Vector object.
   *
   *  @param sType The user defined type from the DTD.
   *  @return A Vector containing the elements as Strings.
   */
  private Vector makeUserTypeEnum(String sType) {
    Vector vEnum = new Vector();
    String sElem = "";

    for (StringTokenizer st = new StringTokenizer(sType, "|"); 
	 st.hasMoreElements(); ) {
      sElem = st.nextToken().trim();
      vEnum.addElement(sElem);
    }

    return vEnum;
  }

  /**
   *  Override toString().
   */
  public String toString() {
    String sOut = "<!ATTLIST " + sElementName + " ";
    for (int i = 0; i < vAttlist.size(); i++) {
      sOut += ((Attribute)vAttlist.elementAt(i)).toString();
    }
    sOut += "\n>\n";

    return sOut;
  }
  
}




