/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

import java.util.*;

public class Attribute {
  /**
   *  Attribute Type: User defined enumerated type
   */
  public final static int USER_DEFINED = 0;

  /**
   *  Attribute Type: CDATA
   */
  public final static int CDATA = 1;

  /**
   *  Attribute Type: ID
   */
  public final static int ID = 2;

  /**
   *  Attribute Type: IDREF
   */
  public final static int IDREF = 3;

  /**
   *  Attribute Type: NOTATION
   */
  public final static int NOTATION = 4;

  /**
   *  Attribute Type: ENTITY
   */
  public final static int ENTITY = 5;

  /**
   *  Attribute Type: ENTITIES
   */
  public final static int ENTITIES = 6;

  /**
   *  Attribute Type: NMTOKEN
   */
  public final static int NMTOKEN = 7;

  /**
   *  Attribute Type: NMTOKENS
   */
  public final static int NMTOKENS = 8;

  /**
   *  Attribute Declaration: #IMPLIED
   */
  public final static int IMPLIED = 0;

  /**
   *  Attribute Declaration: #REQUIRED
   */
  public final static int REQUIRED = 1;

  /**
   *  Attribute Declaration: #FIXED
   */
  public final static int FIXED = 2;

  /**
   *  Parameter name.
   */
  private String sName;

  /**
   *  Attribute type.
   */
  private int iType;

  /**
   *  Attribute declaration.
   */
  private int iDecl;

  /**
   *  Attribute default value.
   */
  private String sValue;

  /**
   *  Enumerated type. Each type (element) is a String.
   */
  private Vector vEnum;

  /**
   *  Creates a new Attribute object. Part of Attlist.
   *
   *  @param paramName The parameter name.
   *  @param type The attribute type.
   *  @param decl The attribute declaration.
   *  @param value The default value.
   *  @param vEnum The user defined enumeration (or NOTATION types).
   */
  public Attribute(String paramName, int type, int decl, String value, Vector vEnum) {
    this.sName = paramName;
    this.iType = type;
    this.iDecl = decl;
    this.sValue = value;
    this.vEnum = vEnum;
  }

  /**
   *  Get the parameter name.
   */
  public String getName() {
    return sName;
  }

  /**
   *  Get the attribute type.
   */
  public int getType() {
    return iType;
  }

  /**
   *  Get the attribute declaration.
   */
  public int getDeclaration() {
    return iDecl;
  }

  /**
   *  Get the attribute default value.
   */
  public String getValue() {
    return sValue;
  }

  /**
   *  Get the user defined enumeration.
   */
  public Vector getEnum() {
    return vEnum;
  }

  /**
   *  Compares two Attribute objects for equality.
   *
   *  @param a The attribute object to be compared to this.
   *  @return True if the objects are the same, false otherwise.
   */
  public boolean equals(Attribute a) {
    if (!sName.equals(a.getName())) {
      return false;
    }

    if (iType != a.getType()) {
      return false;
    }

    if (iDecl != a.getDeclaration()) {
      return false;
    }

    if (!sValue.equals(a.getValue())) {
      return false;
    }

    // won't check for redefinitions of enumerated types
    // technically, I should only need to check for the name, but eh...

    return true;
  }
   
  /**
   *  Override toString().
   */
  public String toString() {
    String sOut = "\n\t" + sName + " " + getTypeString() + getModeString();
    if (sValue.length() > 0) {
      sOut += " \"" + sValue + "\"";
    }

    return sOut;
  }

  /**
   *  For outputting DTD-style attlists. Gets the attribute type.
   */
  public String getTypeString() {
    String sOut = "";
    if (iType == USER_DEFINED) {
      sOut = getEnumString();
    } else if (iType == NOTATION) {
      sOut = "NOTATION " + getEnumString();
    } else if (iType == CDATA) {
      sOut = "CDATA";
    } else if (iType == ID) {
      sOut = "ID";
    } else if (iType == IDREF) {
      sOut = "IDREF";
    } else if (iType == ENTITY) {
      sOut = "ENTITY";
    } else if (iType == ENTITIES) {
      sOut = "ENTITIES";
    } else if (iType == NMTOKEN) {
      sOut = "NMTOKEN";
    } else if (iType == NMTOKENS) {
      sOut = "NMTOKENS";
    }

    return sOut + " ";
  }

  /**
   *  Get the enumeration as a string.
   */
  public String getEnumString() {
    String sOut = "";

    if (vEnum.size() > 0) {
      sOut = "(";
      
      for (int i = 0; i < vEnum.size(); i++) {
	sOut += ((String)vEnum.elementAt(i));
	if (i + 1 < vEnum.size()) {
	  sOut += "|";
	}
      }
      
      sOut += ")";
    }

    return sOut;
  }

  /**
   *  Get the mode string.
   */
  public String getModeString() {
    String sOut = "";

    if (iDecl == IMPLIED) {
      sOut = "#IMPLIED ";
    } else if (iDecl == REQUIRED) {
      sOut = "#REQUIRED ";
    } else if (iDecl == FIXED) {
      sOut = "#FIXED ";
    }
    return sOut;
  }

}

