/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

import java.util.*;

/**
 *  Implements a tree representation of a DTD.
 */
public class DTD {
  /**
   *  An unsorted list of all nodes in the DTD. Intended for searching.
   */
  private Vector vAllNodes;
  
  /**
   *  The element hashtable.
   *  Key is lowercased element name.
   *  Value is the Element object.
   */
  private Hashtable htElements;

  /**
   *  The entity hashtable.
   *  Key is lowercased entity name.
   *  Value is the Entity object.
   */
  private Hashtable htEntities;

  /**
   *  Creates a new DTD.
   */
  public DTD() {
    vAllNodes = new Vector();
    htElements = new Hashtable();
    htEntities = new Hashtable();
  }
  
  /**
   *  Adds an Element to the DTD.
   *
   *  This method does the magic that actually adds the element to 
   *  the DTD, asks the tree to rearrange itself so that child nodes 
   *  are under their parents, rewrites nodes as needed, etc.
   *
   *  Collisions should replace the existing node in the Elements
   *  hashtable and the tree.
   *
   *  @param element The Element object to be added.
   */
  public synchronized void addElement(Element element) {
    // first, add the Element to the all-node vector
    vAllNodes.addElement(element);
    
    // add the element to the hashtable
    htElements.put(element.getName().toLowerCase(), element);
    
    // connect the node into the tree and existing tree nodes
    // into the element
    // *** TODO -- decide if this is needed, or Hashtable and structure
    //     will handle all this well enough ***

    // walk through the existing elements
    //   at each element, walk through the element's optionlists
    //     at each option, resolve the option against an element if null
    Element elem;

    for (Enumeration en = htElements.elements(); en.hasMoreElements(); ) {
      elem = (Element)en.nextElement();

      // check the element's option list
      if (!elem.isEmpty()) {
	validateOption(elem.getOptionList());
      }
    }
  }

  /**
   *  Get an Element from the DTD based on name.
   *
   *  @param name The element name to retrieve.
   *  @return The Element object, or null if not found.
   */
  public Element getElementByName(String name) {
    return (Element)htElements.get(name.toLowerCase());
  }

  /**
   *  Get an enumeration of all the elements in the DTD.
   *
   *  @return An Enumeration of all the elements in the DTD object.
   */
  public Enumeration getElements() {
    return htElements.elements();
  }

  /**
   *  Adds an Attlist to the DTD.
   *
   *  This method associates the attribute list with the correct element.
   *
   *  Note that if an attlist is added BEFORE the element it references,
   *  it will NOT be associated and will be lost.
   *
   *  @param attlist The Attlist object to be added
   */
  public void addAttlist(Attlist attlist) {
    // get the element name from the attlist
    String sElemName = attlist.getElementName();

    // get the Element object
    Element elem = getElementByName(sElemName);

    // does the element already have an attribute list associated with it?
    if (elem.getAttlist() != null) {
      // yes...merge the attribute lists
      elem.getAttlist().merge(attlist);
    } else {
      // no...set the attribute lists
      elem.setAttlist(attlist);
    }
  }
  
  /**
   *  Adds an entity to the DTD.
   *
   *  This method adds the entity tag to the DTD. Because this is a 
   *  non-validating parser, the entity is only stored and not properly 
   *  expanded.
   *
   *  @param entity The Entity object to be added.
   */
  public void addEntity(Entity entity) {
    htEntities.put(entity.getName().toLowerCase(), entity);
  }

  /**
   *  Get the entities from the entity tree.
   *
   *  @return The Entity objects in an Enumeration.
   */
  public Enumeration getEntities() {
    return htEntities.elements();
  }

  /**
   *  Get an Entity by name.
   *
   *  @param name The entity name.
   *  @return The Entity object matching the name, or null if not found.
   */
  public Entity getEntityByName(String name) {
    return (Entity)htEntities.get(name.toLowerCase());
  }

  /**
   *  Validates the OptionList for the Element (and all nested OptionList
   *  objects).
   *
   *  <UL>Walk the OptionList
   *    <LI>At each element:
   *    <UL>
   *      <LI>If it's an OptionList, recurse</LI>
   *      <LI>If it's an Option, check to see if the Element is null...set it
   *          if needed.</LI>
   *    </UL>
   *    </LI>
   *  </UL>
   *
   *  @param list The OptionList to consider.
   */
  private void validateOption(OptionList list) {
    Vector v = list.getOptions();
    Option o;

    for (int i = 0; i < v.size(); i++) {
      if (v.elementAt(i) instanceof OptionList) {
	validateOption((OptionList)v.elementAt(i));
      } else if (v.elementAt(i) instanceof Option) {
	o = (Option)v.elementAt(i);

	// only attempt to resolve the Option's element if it's null
	if (o.getElement() == null) {
	  // resolve this as best as possible
	  // this will be set to null (automatically) if we fail the lookup
	  o.setElement(getElementByName(o.getName()));
	}
      } else {
	System.err.println("Unexpected type in OptionList " + list);
      }
    }
  }
}


