/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

public class Element
{
  /**
   *  The element name.
   */
  private String sElementName;
  
  /**
   *  The element content from the DTD.
   */
  private String sElementContent;
  
  /**
   *  The elements content (OptionList)
   */
  private OptionList optionList;

  /**
   *  The attribute list.
   */
  private Attlist attlist;
  
  /**
   *  Creates a new element.
   *
   *  @param elementName The element name.
   *  @param elementContent The element content from the DTD.
   */
  public Element(String elementName, String elementContent) {
    this.sElementName = elementName;
    this.sElementContent = elementContent;
    
    // turn the element content into the proper representation
    optionList = Element.makeContent(sElementContent);

    // For debugging
    //    System.out.println(toString());
  }

  /**
   *  Creates an EMPTY element.
   *
   *  @param elementName The element name.
   */
  public Element(String elementName) {
    this.sElementName = elementName;

    this.optionList = null;
  }
  
  /**
   *  Get the contents/OptionList.
   *
   *  @return The option list structure, or null if this is an EMPTY element.
   */
  public OptionList getOptionList() {
    return optionList;
  }

  /**
   *  Indicates if this is an EMPTY element.
   *
   *  @return True is this is an EMPTY element, false otherwise.
   */
  public boolean isEmpty() {
    return optionList == null;
  }

  /**
   *  Get the element's name.
   *
   *  @return The element's name.
   */
  public String getName() {
    return sElementName;
  }

  /**
   *  Get the attribute list.
   *
   *  @return The attribute list for this element.
   */
  public Attlist getAttlist() {
    return attlist;
  }

  /**
   *  Set the attribute list.
   *
   *  @param attlist The attribute list.
   */
  public void setAttlist(Attlist attlist) {
    this.attlist = attlist;

    // set the attlist's element
    this.attlist.setElement(this);
  }
  
  /**
   *  Turn the supplied content string from the DTD into
   *  a proper, in-memory representation.
   */
  private static OptionList makeContent(String sContent) {
    // the content string should be bound by parantheses
    // remove the leading and trailing characters
    sContent = sContent.substring(1, sContent.length() - 1);

    // the way to get the elements is to...
    // get an opening paren...
    // walk until you find:
    //   - a new opening paren -- recurse into
    //   - an option -- get the cardinality
    //   - a sequence or option indicator (',' or '|')
    //   - a closing paren -- pop
    // the option is defined as...
    //   - an new option (text)
    //   - everything between nested parens
    
    OptionList ol = new OptionList();
    
    String sOptionName = "";
    String sSubOption = "";
    
    for (int i = 0; i < sContent.length(); i++) {
      if (sContent.charAt(i) == '(') {
	// open paren found
	// find the closing parent
	int parenCount = 1;
	int parenStart = i;
	
	// scan the next character...
	i++;
	
	// find the end of the sub-option list
	while (i < sContent.length() && parenCount > 0) {
	  if (sContent.charAt(i) == '(') {
	    parenCount++;
	  } else if (sContent.charAt(i) == ')') {
	    parenCount--;
	  }
	  
	  i++;
	}
	
	// make the sub-option list
	if (i <= sContent.length()) {
	  sSubOption = sContent.substring(parenStart, i).trim();
	  if (sSubOption.length() > 0) {
	    ol.addOption(Element.makeContent(sSubOption));
	  }

	  // decrement the character we're examining -- otherwise,
	  // the previous i++ will cause the loop to skip the
	  // character following the closed parentheses
	  i--;
	}
      } else if (sContent.charAt(i) == '|') {
	// check to see if the option name was set
	sOptionName = sOptionName.trim();
	if (sOptionName.length() > 0) {
	  // add the option to the option list
	  ol.addOption(new Option(sOptionName));
	  sOptionName = "";
	}
	// set the overall option list type
	ol.setType(OptionList.LIST_OR);
      } else if (sContent.charAt(i) == ',') {
	// check to see if the option name was set
	sOptionName = sOptionName.trim();
	if (sOptionName.length() > 0) {
	  // add the option to the option list
	  ol.addOption(new Option(sOptionName));
	  sOptionName = "";
	}
	// set the overall option list type
	ol.setType(OptionList.LIST_AND);
      } else if (sContent.charAt(i) == '+') {
	// check to see if the option name was set
	sOptionName = sOptionName.trim();
	if (sOptionName.length() > 0) {
	  // add the option to the option list
	  ol.addOption(new Option(sOptionName));
	  sOptionName = "";
	}

	// set the last option to be a one-to-many type
	ol.setLastOptionType(Option.OPTION_ONE_TO_MANY);
      } else if (sContent.charAt(i) == '?') {
	// check to see if the option name was set
	sOptionName = sOptionName.trim();
	if (sOptionName.length() > 0) {
	  // add the option to the option list
	  ol.addOption(new Option(sOptionName));
	  sOptionName = "";
	}
	// set the last option to be optional
	ol.setLastOptionType(Option.OPTION_OPTIONAL);
      } else if (sContent.charAt(i) == '*') {
	// check to see if the option name was set
	sOptionName = sOptionName.trim();
	if (sOptionName.length() > 0) {
	  // add the option to the option list
	  ol.addOption(new Option(sOptionName));
	  sOptionName = "";
	}
	// set the last element to be none-to-many type
	ol.setLastOptionType(OptionList.OPTION_NONE_TO_MANY);
      } else {
	// add the character to the option name
	sOptionName += sContent.substring(i, i + 1);
      }
    } // end for

    // check to see if the option name was set
    sOptionName = sOptionName.trim();
    if (sOptionName.length() > 0) {
      // add the option to the option list
      ol.addOption(new Option(sOptionName));
      sOptionName = "";
    }

    // return the option list
    return ol;
  }

  /**
   *  Create a String representation of the contents of this Element 
   *  and all its aggregate objects.
   *
   *  @return A String representing the contents of this object.
   */
  public String toString() {
    String sBody = "";

    if (isEmpty()) {
      sBody = sElementName + " EMPTY";
    } else {
      sBody = sElementName + " " + optionList;
    }

    return "<!ELEMENT " + sBody + ">\n" + 
      (attlist != null ? attlist.toString() : "");
  }
}
