/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

import java.io.*;
import java.net.*;
import java.util.*;

public class Entity
{
  /**
   *  Indicator that this is NOT an external entity type.
   */
  public final static int EXT_NONE = 0;

  /**
   *  The SYSTEM external entity type.
   */
  public final static int EXT_SYSTEM = 1;

  /**
   *  The PUBLIC external entity type.
   */
  public final static int EXT_PUBLIC = 2;

  /**
   *  The parent DTD for this entity. Required for parsing external
   *  entities.
   */
  private DTD dtd;
  
  /**
   *  The entity name.
   */
  private String sEntityName;
  
  /**
   *  The entity contents (from the DTD).
   */
  private String sEntityContents;

  /**
   *  The entity value (if type is EXT_NONE).
   */
  public String sValue = "";

  /**
   *  The external entity type (SYSTEM or PUBLIC).
   */
  private int iExtType;

  /**
   *  The public ID (if iExtType is PUBLIC).
   */
  private String sPubId = "";

  /**
   *  The URI (if any).
   */
  private String sURI = "";

  /**
   *  The notation name, if any.
   */
  private String sNotation = "";

  /**
   *  Indicates that this Entity object should attempt to resolve
   *  external references.
   */
  private boolean bResolve = false;
  
  /**
   *  String that indicates a notation name (NDATA).
   */
  private final static String NDATA = "NDATA";

  /**
   *  String that indicates a SYSTEM external entity type.
   */
  private final static String SYSTEM = "SYSTEM";

  /**
   *  String that indicates a PUBLIC external entity type.
   */
  private final static String PUBLIC = "PUBLIC";

  /**
   *  Creates a new Entity from the DTD data.
   *
   *  @param entityName The entity name from the DTD.
   *  @param entityContents The entity contents from the DTD.
   *  @param dtd The DTD.
   *  @param bResolve The Entity should resolve external entities.
   */
  public Entity(String entityName, String entityContents, DTD dtd, 
		boolean bResolve) {
    this.sEntityName = entityName;
    this.sEntityContents = entityContents;
    this.dtd = dtd;
    this.bResolve = bResolve;

    // parse the DTD entity contents
    parseContents(entityContents);
  }

  /**
   *  Get the entity's name.
   *
   *  @return A String containing the entity name.
   */
  public String getName() {
    return sEntityName;
  }

  /**
   *  Indicates the type of entity:  EXT_NONE, EXT_SYSTEM or EXT_PUBLIC.
   *
   *  @return An integer indicating EXT_NONE, EXT_SYSTEM, or EXT_PUBLIC.
   */
  public int getExternalType() {
    return iExtType;
  }

  /**
   *  Indicates that this is a NDATA entity.
   *
   *  @return True if these has notation data.
   */
  public boolean isNdata() {
    return sNotation.length() > 0;
  }

  /**
   *  Get the value of the entity.
   *
   *  @return A String containing the value.
   */
  public String getValue() {
    return sValue;
  }

  /**
   *  Get the public id of the entity.
   *
   *  @return A String containing the public id.
   */
  public String getPublicId() {
    return sPubId;
  }

  /**
   *  Get the URI of the entity.
   *
   *  @return A String containing the URI.
   */
  public String getUri() {
    return sURI;
  }

  /**
   *  Get the notation name.
   *
   *  @return A String containing the notation name.
   */
  public String getNotation() {
    return sNotation;
  }

  /**
   *  Parses the entity contents from the DTD.
   *
   *  @param sContents The String from the DTD defining the entity contents.
   */
  private void parseContents(String sContents) {
    // Already have the entity name stored. The first word will
    // indicate what type of entity this is.
    sContents = sContents.trim();
    int i = sContents.indexOf(" ");
    int j = 0;

    if (i < 0) {
      // one word...it's DEFINITELY a value type
      sValue = stripQuotes(sContents);
    } else {
      // more that one word
      String sWord = sContents.substring(0, i);

      // check the word
      if (sWord.toUpperCase().equals(PUBLIC)) {
	// set the public type
	iExtType = EXT_PUBLIC;

	// we need to have two more elements...
	// public id and uri
	i = sContents.indexOf("\"");
	j = sContents.indexOf("\"", i + 1);
	sPubId = stripQuotes(sContents.substring(i + 1, j));

	i = sContents.indexOf("\"", j + 1);
	j = sContents.indexOf("\"", i + 1);
	sURI = stripQuotes(sContents.substring(i + 1, j));

	if (bResolve) {
	  parseExternal();
	}
      } else if (sWord.toUpperCase().equals(SYSTEM)) {
	// set the system type
	iExtType = EXT_SYSTEM;

	// we will have one or three more elements...
	// uri is first
	i = sContents.indexOf("\"");
	j = sContents.indexOf("\"", i + 1);
	sURI = stripQuotes(sContents.substring(i + 1, j));

	// check to see if more elements are present
	if (j + 1 < sContents.length()) {
	  // skip the next word...it's NDATA
	  i = sContents.indexOf(NDATA, j);
	  j = sContents.indexOf(" ", i);
	  sNotation = sContents.substring(j).trim();
	}

	if (bResolve) {
	  parseExternal();
	}
      } else {
	System.err.println("Unknown Entity external type: '" + sWord + "'");
      }
    }
  }

  /**
   *  Strip quotes from a string.
   *
   *  @param s Input string.
   */
  private String stripQuotes(String s) {
      // strip leading and trailing quotes
      if (s.startsWith("\"")) {
	s = s.substring(1);
      }

      if (s.endsWith("\"")) {
	s = s.substring(0, s.length() - 1);
      }

      return s;
  }

  /**
   *  Parse an external Entity. Brings those elements into the DTD.
   */
  private void parseExternal() {
    // check to see if we got a real URL or a file reference
    boolean bURL = false;
    URL url = null;
    BufferedReader br = null;    

    try {
      url = new URL(sURI);
      bURL = true;
    } catch (MalformedURLException e) {
      // not a real URL
    }
    
    try {
      if (bURL) {
	// open the URL for reading
	URLConnection uc = url.openConnection();
	InputStream is = uc.getInputStream();
	// get a buffered reader from the URI
	br = new BufferedReader(new InputStreamReader(is));
      } else {
	// get a buffered read for file input
	br = new BufferedReader(new FileReader(sURI));
      }

      // scan the input file, put the elements in the dtd
      Parser.scan(dtd, br);
    } catch (Exception e) {
      // catchall exception handler
      System.err.println("Error parsing external entity in DTD. Entity is:\n" +
			 toString() + "Exception: " + e.getMessage());
      e.printStackTrace();
    }
  }

  /**
   *  Show the ENTITY in DTD format.
   */
  public String toString() {
    String sOut = "<!ENTITY " + (bResolve ? "% " : "") + sEntityName;

    switch (iExtType) {
    case EXT_NONE:
      sOut += " \"" + sValue + "\"";
      break;
    case EXT_SYSTEM:
      sOut += " " + SYSTEM + " \"" + sURI + "\"";
      if (sNotation.length() > 0) {
	sOut += " " + NDATA + " " + sNotation;
      }
      break;
    case EXT_PUBLIC:
      sOut += " " + PUBLIC + " \"" + sPubId + "\" \"" +sURI + "\"";
      break;
    default:
      sOut += "<incorrect external entity type " + iExtType + ">";
    }

    return sOut + ">\n";
  }
}


