/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

public class Option {
  /**
   *  The name of the option.
   */
  private String sOptionName;

  /**
   *  The option type.
   */
  private int iOptionType;
  
  /**
   *  The Element associated with this option.
   */
  private Element element; 

  /**
   *  Indicates that the option cardinality is not specified.
   */
  public final static int OPTION_NONE = 0;

  /**
   *  Indicates that the option cardinality is none-to-many (*)
   */
  public final static int OPTION_NONE_TO_MANY = 1;

  /**
   *  Indicates that the option cardinality is optional (?)
   */
  public final static int OPTION_OPTIONAL = 2;
  
  /**
   *  Indicates that the option cardinality is one-to-many (+)
   */
  public final static int OPTION_ONE_TO_MANY = 3;

  /**
   *  Creates a new Option object.
   *
   *  @param optionname The name of this option (from the DTD).
   */
  public Option(String optionname) {
    this.sOptionName = optionname;
  }

  /**
   *  Creates a new Option object.
   */
  public Option() {
    // do nothing
  }

  /**
   *  Get the option name.
   *
   *  @return The option name.
   */
  public String getName() {
    return sOptionName;
  }

  /**
   *  Gets the associated Element for this option.
   *
   *  @return The Element object, or none if not yet associated.
   */
  public Element getElement() {
    return element;
  }

  /**
   *  Sets the Element associated with this option.
   *
   *  @param element The associated Element object.
   */
  public void setElement(Element element) {
    this.element = element;
  }

  /**
   *  Gets the type (cardinality) of this option. The type is one of
   *  OPTION_NONE_TO_MANY, OPTION_OPTIONAL, OPTION_ONE_TO_MANY, or
   *  OPTION_NONE.
   *
   *  @return An integer representing the cardinality of this option.
   */
  public int getOptionType() {
    return iOptionType;
  }

  /**
   *  Sets the type (cardinality) of this option. The type is one of
   *  OPTION_NONE_TO_MANY, OPTION_OPTIONAL, OPTION_ONE_TO_MANY, or
   *  OPTION_NONE.
   *
   *  @param optiontype The integer option type.
   */
  public void setOptionType(int optiontype) {
    this.iOptionType = optiontype;
  }

  /**
   *  Returns a String representation of this Option.
   *
   *  @return A string representation of this Option object.
   */
  public String toString() {
    String sOut = sOptionName;
    switch (iOptionType) {
    case OPTION_NONE:
      sOut += "";
      break;
    case OPTION_NONE_TO_MANY:
      sOut += "*";
      break;
    case OPTION_OPTIONAL:
      sOut += "?";
      break;
    case OPTION_ONE_TO_MANY:
      sOut += "+";
      break;
    default:
      sOut += "<Incorrect setting: " + iOptionType + ">";
      break;
    }

    return sOut;
  }
}


