/*
DTD parser library for Java.
Copyright (C) 2000 Christopher R. Jones

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package org.menagery.dtd;

import java.util.*;

public class OptionList extends Option {
  /**
   *  The last option added to the list. Used when setting attributes.
   */
  private Option oLastOptionAdded;

  /**
   *  The vector (in order) of all options added to this OptionList.
   */
  private Vector vOptions;

  /**
   *  The list type (AND or OR).
   */
  private int iListType;

  /**
   *  Indicates that the list is an exclusive OR type list (|)
   */
  public final static int LIST_OR = 1;

  /**
   *  Indicates that the list is a sequential AND type list (,)
   */
  public final static int LIST_AND = 0;

  /**
   *  Creates a new OptionList object.
   */
  public OptionList() {
    vOptions = new Vector();
    
    // by default list type is AND
    iListType = LIST_AND;
  }

  /**
   *  Gets the list type.
   *
   *  @return The integer type, either LIST_OR or LIST_AND.
   */
  public int getType() {
    return iListType;
  }

  /**
   *  Sets the list type.
   *
   *  @param type The integer list type, either LIST_OR or LIST_AND.
   */
  public void setType(int type) {
    this.iListType = type;
  }

  /**
   *  Get the option list.
   *
   *  @return The list of options as a Vector.
   */
  public Vector getOptions() {
    return vOptions;
  }

  /**
   *  Add an Option to the OptionList.
   *
   *  @param option The Option to be added.
   */
  public void addOption(Option option) {
    vOptions.addElement(option);
    oLastOptionAdded = option;
  }

  /**
   *  Sets the type (cardinality) of the last option added (Option or 
   *  OptionList). The type is one of OPTION_NONE_TO_MANY, OPTION_OPTIONAL,
   *  OPTION_ONE_TO_MANY, or OPTION_NONE.
   *
   *  @param optiontype The integer option type.
   */
  public void setLastOptionType(int optiontype) {
    oLastOptionAdded.setOptionType(optiontype);
  }

  /**
   *  Returns a string representation of this object and it's
   *  aggregate objects.
   *
   *  @return A String containing a representation of this object.
   */
  public String toString()
  {
    String sOut = "(";

    String delim = (iListType == LIST_OR ? "|" : ",");

    for (int i = 0; i < vOptions.size(); i++) {
      if (vOptions.elementAt(i) instanceof OptionList) {
	sOut += ((OptionList)vOptions.elementAt(i)).toString();
      } else if (vOptions.elementAt(i) instanceof Option) {
	sOut += ((Option)vOptions.elementAt(i)).toString();
      } else {
	sOut += vOptions.elementAt(i).toString();
      }

      if (i < vOptions.size() - 1) {
	sOut += delim;
      }
    }

    sOut += ")";

    // show the cardinality
    switch (getOptionType()) {
    case Option.OPTION_NONE:
      break;
    case Option.OPTION_NONE_TO_MANY:
      sOut += "*";
      break;
    case Option.OPTION_OPTIONAL:
      sOut += "?";
      break;
    case Option.OPTION_ONE_TO_MANY:
      sOut += "+";
      break;
    default:
      sOut += "<Invalid cardinality " + getOptionType() + ">";
      break;
    }

    return sOut;
  }

}




