/*  ConnectionHandler.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 *  Responsible for keeping track of connection info objects, checking
 *  for new input on connections, and passing new output to
 *  connections.
 *  <p>
 *  Is similar to the functionality of the UNIX select() or poll() call.
 *  <p>
 *  Runs in a thread to allow network I/O to operate asynchronously
 *  from the rest of the system. Network I/O is checked for input
 *  first (on each connection), then attempts to empty the output
 *  queue.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class ConnectionHandler implements Runnable {
    /**
     *  The sleep time for this thread (in between processing runs).
     */
    protected final static int SLEEP_MILLIS = 100;

    /**
     *  Registered connection info objects.
     */
    protected HashMap hmConnectionInfo;

    /**
     *  The input (incoming message) queue.
     */
    protected InputQueue qInput;

    /**
     *  The output (outgoing message) queue.
     */
    protected Queue qOutput;

    /**
     *  The handler thread.
     */
    protected Thread tHandler;

    /**
     *  Flag to indicate if this is currently handling I/O.
     *
     *  Set this to false to stop handling I/O and shutdown all the
     *  current connections.
     */
    protected boolean bHandling;

    /**
     *  Create a new connection handler.
     *
     *  Do the housekeeping to ensure this is set up correctly.
     */
    public ConnectionHandler(InputQueue qInput, Queue qOutput) {
        this.qInput = qInput;
        this.qOutput = qOutput;
        
        hmConnectionInfo = new HashMap();

        // start the network I/O processor
        tHandler = new Thread(this);
        tHandler.start();
    }

    /**
     *  Register a new connection info object.
     *  <p>
     *  This isn't synchronized because this should only be called
     *  from the new connection info object.
     */
    public void registerConnectionInfo(ConnectionInfo info) {
        hmConnectionInfo.put(info.getConnectionId(), info);
    }
    
    /**
     *  Unregister a connection info object.
     *  <p>
     *  This is synchronized (internally) so that it won't be 
     *  unregistered twice at once (unregister may be called from 
     *  more than one place) and so that the connection won't be used
     *  while it's being closed.
     */
    public void unregisterConnectionInfo(ConnectionInfo info) {
        // remove the connection info from the registered set
        synchronized (hmConnectionInfo) {
            hmConnectionInfo.remove(info.getConnectionId());
        }

        try {
            // flush the output buffer
            info.bos.flush();
        } catch (IOException e) {
            // ignore any errors as this is shutting down anyway
        }

        // terminate the connection
        info.terminate();
    }

    /**
     *  Indicates if this is handling I/O.
     *
     *  @return true if this is handling connection I/O.
     */
    public boolean isHandling() {
        return bHandling && tHandler != null;
    }

    /**
     *  Stops the handler from handling I/O.
     *  <p>
     *  Create a new handler to start handling I/O again. This will
     *  disconnect all the current clients.
     */
    public void stopHandler() {
        bHandling = false;
    }

    /**
     *  Process connection I/O, including checking for available
     *  messages from the client and from the message handler.
     */
    public void run() {
        bHandling = true;

        BufferedInputStream bis;
        BufferedOutputStream bos;
        Iterator it;
        ConnectionInfo info;
        int iAvail;
        boolean bError = false;
        byte [] baMsg;
        Message msg;

        while (bHandling) {
            // first, check for available bytes on each connection
            synchronized (hmConnectionInfo) {
                for (it = hmConnectionInfo.values().iterator(); it.hasNext(); ) {
                    // get the next connection
                    info = (ConnectionInfo)it.next();
                    // get the input stream from the connection
                    bis = info.getInputStream();
                    // get the available bytes from the input stream
                    try {
                        iAvail = bis.available();
                        info.resetError();
                    } catch (IOException e) {
                        // problem getting available bytes
                        bError = true;
                        iAvail = 0;
                    }
                        
                    // check the available bytes
                    if (iAvail > 0) {
                        // pass the incoming bytes to the input queue
                        baMsg = new byte[iAvail];
                        
                        try {
                            bis.read(baMsg, 0, iAvail);
                            qInput.incomingMessage(
                              info.getConnectionId(), baMsg);
                            info.resetError();
                        } catch (IOException e) {
                            // problem getting bytes from connection
                            bError = true;
                        } catch (MessageTooLargeException e) {
                            // message was too large
                            System.err.println(e.getMessage());
                            bError = true;
                        }
                    }
                }
            }

            // ...so these can be garbage collected, in case it runs
            // before we're done here
            bis = null;
            it = null;
            info = null;

            // flush the output queue
            synchronized (qOutput) {
                while (qOutput.getSize() > 0) {
                    // get the next message
                    try {
                        msg = (Message)qOutput.dequeue();
                    } catch (NoSuchElementException e) {
                        // this should happen, but the queue was empty
                        System.err.println("ASSERT FAILED: Queue empty when it shouldn't be!");
                        break;
                    }
                    // get the connection associated with the message
                    info =
                    (ConnectionInfo)hmConnectionInfo.get(msg.getConnectionId());
                    // get the output stream
                    bos = info.getOutputStream();
                    // get the byte representation of the message
                    baMsg = msg.getBytes();
                    // write the message to the output stream
                    try {
                        bos.write(baMsg, 0, baMsg.length);
                        bos.flush();
                        info.resetError();
                    } catch (IOException e) {
                        // problem writing bytes to connection
                        bError = true;
                    }
                }
            }

            // ...so these can be garbage collected, in case it runs
            // while we're asleep
            bos = null;
            msg = null;
            info = null;

            // sleep for a fraction of a second
            try {
                tHandler.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // ignore this
            }
        }
    }
}

/*
 * $Log$
 */
