/*  InputQueueTester.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.util.NoSuchElementException;

/**
 *  Test critical InputQueue functionality.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class InputQueueTester {
    public static void main(String [] args) {
        // create a new InputQueue
        InputQueue q = new InputQueue();

        // test the queue functionality
        Object o1a = new Object();
        q.enqueue(o1a);
        Object o2a = new Object();
        q.enqueue(o2a);
        Object o3a = new Object();
        q.enqueue(o3a);
        Object o1b = q.dequeue();
        Object o2b = q.dequeue();
        Object o3b = q.dequeue();
        if (o1b == o1a && o2b == o2a && o3b == o3a) {
            System.err.println("OK Queueing worked");
        }

        try {
            q.dequeue();
        } catch (NoSuchElementException e) {
            System.err.println("OK Queue threw exception when empty.");
        }

        q.enqueue(o1a);
        q.enqueue(o2a);
        q.enqueue(o3a);
        if (q.peek().equals(o1a)) {
            System.err.println("OK Peeking working");
        }

        System.err.println("Queue size should be 3: " +
          String.valueOf(q.getSize()));

        // byte array tests
        // simple message
        String sMessage = Message.HEADER + "this is my test message" +
                          Message.FOOTER;
        byte [] baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("0", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown! " + e.getMessage());
        }
        
        System.err.println("Queue size should be 4: " +
          String.valueOf(q.getSize()));
        q.dequeue();
        q.dequeue();
        q.dequeue();
        System.err.println("Should say:  0: 'this is my test message'");
        System.err.println(((Message)q.dequeue()).toString());
        
        // message with garbage at beginning
        sMessage = "foo" + Message.HEADER + "this is my second test message" + Message.FOOTER;
        baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("0", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown!" + e.getMessage());
        }
        System.err.println("Should say: 0: 'this is my second test message'");
        System.err.println((Message)q.dequeue());

        // message with garbage at end
        sMessage = Message.HEADER + "this is my third test message" +
        Message.FOOTER + "bar";
        baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("0", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown!" + e.getMessage());
        }
        System.err.println("Should say: 0: 'this is my third test message'");
        System.err.println((Message)q.dequeue());

        // message for another connection in the middle
        sMessage = Message.HEADER + "this is a second connection test" + Message.FOOTER;
        baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("1", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown!" + e.getMessage());
        }
        System.err.println("Should say: 1: 'this is a second connection test'");
        System.err.println((Message)q.dequeue());

        // normal message with a partial message at end
        sMessage = Message.HEADER + "this is my fourth test message" +
        Message.FOOTER + Message.HEADER + "this is my fifth";
        baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("0", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown!" + e.getMessage());
        }
        System.err.println("Should say: 0: 'this is my fourth test message'");
        System.err.println((Message)q.dequeue());
        
        // completion of incomplete message
        sMessage = " test message" + Message.FOOTER;
        baMessage = sMessage.getBytes();
        try {
            q.incomingMessage("0", baMessage);
        } catch (MessageTooLargeException e) {
            System.err.println("This message should not have been thrown!" + e.getMessage());
        }
        System.err.println("Should say: 0: 'this is my fifth test message'");
        System.err.println((Message)q.dequeue());
       
        // finally, test the listener
        Queue qOut = new Queue();
        ConnectionHandler h = new ConnectionHandler(q, qOut);
        Listener l = new Listener(h, 8887);
        Message msg;

/*
        // last test: network busy loop
        // (will consume all your CPU)
        while (true) {
            try {
                msg = (Message)q.dequeue();
                qOut.enqueue(msg);
            } catch (NoSuchElementException e) {
                // do nothing
            }
        }
*/

        // test with the message handler
        MessageHandler mh = new MessageHandler(q, qOut);
        

    }
}
