/*  Listener.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;

/**
 *  Listens for new connections.
 *
 *  Passes connections to the connection handler for lifecycle
 *  management.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class Listener implements Runnable {
    /**
     *  The number of milliseconds to sleep between servicing each new
     *  socket request.
     */
    protected final static int SLEEP_MILLIS = 250;

    /**
     *  The server socket.
     */
    protected ServerSocket server;

    /**
     *  The port on which to listen for new connections.
     */
    protected int iPort;

    /**
     *  The maximum number of un-accepted connections queued.
     */
    protected int BACKLOG = 25;

    /**
     *  The connection handler which gets new socket connections.
     */
    protected ConnectionHandler handler;

    /**
     *  Flag to indicate listening for new connections.
     */
    protected boolean bListening;

    /**
     *  Listening thread.
     */
    Thread tListener;

    /**
     *  Create a new Listener.
     *
     *  Start listening and passing new connections to the connection
     *  handler.
     *
     *  @param handler the connection handler.
     *  @param iPort the port on which to listen.
     */
    public Listener(ConnectionHandler handler, int iPort) {
        this.handler = handler;
        this.iPort = iPort;
        start();
    }

    /**
     *  Indicate if this is listening.
     */
    public boolean isListening() {
        return bListening && tListener != null;
    }

    /**
     *  Start listening.
     */
    public void start() {
        if (tListener == null) {
            tListener = new Thread(this);
            tListener.start();
        }
    }

    /**
     *  Stop listening.
     */
    public void stop() {
        if (tListener != null) {
            bListening = false;
        }

        tListener = null;
    }

    /**
     *  Listen for new connections and pass them to the connection
     *  handler.
     */
    public void run() {
        try {
            server = new ServerSocket(iPort, BACKLOG);
        } catch (IOException e) {
            // report that the server could not be started
            System.err.println("Unable to start server socket: " +
            e.getMessage());
            return;
        }
        
        // set the listening flag true to listen indefinitely
        bListening = true;
        
        // socket connection object
        Socket sock = null;
        
        // connection id
        int iConnectionId = 0;

        while (bListening) {
            try {
                // get the next connection
                sock = server.accept();

                // create the (self-registering) connection info
                new ConnectionInfo(handler, sock, iConnectionId);
            } catch (IOException e) {
                // problem with this connection info so don't pass it
                // to the handler
                System.err.println("Couldn't establish connection info on socket connection from " + 
                  ((sock != null) ? sock.getInetAddress().toString() : 
                   "unknown address") + 
                  ": " + e.getMessage());
            } finally {
                sock = null;
            }

            // increment the connection id
            iConnectionId++;

            try {
                // sleep for a period of time before servicing the
                // next connection
                tListener.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // ignore this exception
            }
        }

        // close the server socket
        try {
            server.close();
        } catch (IOException e) {
            System.err.println("Problem closing the server socket: " +
            e.getMessage());
        } finally {
            server = null;
        }
    }
}
