/*  Message.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

/**
 *  Message object.
 *
 *  Stores a complete message, including the associated originating
 *  connection. Acts as a kind of "network event" that is consumed 
 *  by the message handler, passed to a processor, modified, and
 *  passed back to the output queue where is is consumed by the
 *  connection handler.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class Message {
    /**
     *  The header that indicates the start of a message.
     */
    public final static String HEADER = "<!start!>";

    /**
     *  The header that indicates the end of a message.
     */
    public final static String FOOTER = "<#end#>";

    /**
     *  Connection id.
     *
     *  Used to associate messages with the originating connection or
     *  destination connection.
     */
    protected String sConnectionId;

    /**
     *  Message content.
     *
     *  The complete message content.
     */
    protected String sMessageContent;

    /**
     *  Create a new Message object.
     *
     *  Creates a new Message object. The object will still need to be
     *  injected into the message handler input queue or connection
     *  handler output queue in order to be consumed.
     *
     *  @param sConnectionId the source or destination connection id.
     *  @param sMessageContent the content of the message.
     */
    public Message(String sConnectionId, String sMessageContent) {
        this.sConnectionId = sConnectionId;
        this.sMessageContent = sMessageContent;
    }

    /**
     *  Get the Message's associated connection id.
     *
     *  @return the associated connection id.
     */
    public String getConnectionId() {
        return sConnectionId;
    }

    /**
     *  Set the Message's associated connection id.
     *
     *  @param sConnectionId the source or destination connection id.
     */
    public void setConnectionId(String sConnectionId) {
        this.sConnectionId = sConnectionId;
    }

    /**
     *  Get the Message's content.
     *
     *  @return the message content.
     */
    public String getContent() {
        return sMessageContent;
    }

    /**
     *  Set the Message's content.
     *
     *  @param sMessageContent the message content.
     */
    public void setContent(String sMessageContent) {
        this.sMessageContent = sMessageContent;
    }

    /**
     *  Get the message's byte representation.
     *
     *  @return the message as an array of bytes.
     */
    public byte [] getBytes() {
        return sMessageContent.getBytes();
    }

    /**
     *  Get the message information.
     *
     *  @return a representation of the message.
     */
    public String toString() {
        return new String(sConnectionId + ": '" +
          sMessageContent + "'");
    }
}

/*
 * $Log$
 */
