/*  MessageHandler.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.util.NoSuchElementException;

/**
 *  Responsible for retrieving messages from the input queue, passing
 *  them off to the proper processor thread, and sending responses back to
 *  the output queue.
 *  <p>
 *  Additionally, manages the processor thread pool.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class MessageHandler implements Runnable {
    /**
     *  The time to sleep between fetches from the input queue.
     */
    protected static int SLEEP_MILLIS = 50;
    
    /**
     *  The maximum number of processor threads allowed.
     *  <p>
     *  In effect, the maximum number of commands permitted to execute
     *  at once. This number should be tuned for each installation
     *  based on the number of requests and the type of system
     *  (multi-processor systems or systems that implement Java
     *  threads as microthreads will tend to do better with more
     *  threads than single processor systems or systems that
     *  implement Java threads as system processes).
     *  <p>
     *  This value is passed to the thread pool.
     */
    protected int MAX_PROCESSORS = 8;
    
    /**
     *  The available processor thread pool.
     */
    ThreadPool tpProcessor;

    /**
     *  The thread to fetch messages from the input queue.
     */
    Thread tHandler;

    /**
     *  Indicates if this is checking for new messages on the input
     *  queue.
     */
    boolean bHandling;

    /**
     *  The message input queue.
     */
    Queue qInput;

    /**
     *  The message output queue.
     */
    Queue qOutput;

    /**
     *  Creates a new message handler.
     *
     *  @param qInput the message input queue.
     *  @param qOutput the message output queue.
     */
    public MessageHandler(Queue qInput, Queue qOutput) {
        this.qInput = qInput;
        this.qOutput = qOutput;
        tpProcessor = new ThreadPool(MAX_PROCESSORS);
        tHandler = new Thread(this);
        tHandler.start();
    }

    /**
     *  Shut down the message handler and the message processor thread
     *  pool.
     */
    public void shutdown() {
        // shut down the input queue processor
        bHandling = false;
    }
    
    /**
     *  Check for new messages on the input queue, get a new processor
     *  thread from the processor thread pool, and pass the message to
     *  the processor thread.
     */
    public void run() {
        // the message from the input queue
        Message msg;
        
        // the processor thread
        ProcessorThread pt;
        
        // indicate that this is handling input
        bHandling = true;
        
        while (bHandling) {
            try {
                // get the next message on the input queue
                msg = (Message)qInput.dequeue();

                // get the next free thread from the thread pool
                // (this call blocks)
                pt = tpProcessor.nextFree();
                
                // pass the message to the processor thread
                pt.process(msg, qOutput);
            } catch (NoSuchElementException e) {
                // ignore this...there is no message to pass to the
                // processor
            }

            try {
                // sleep for a while
                tHandler.sleep(SLEEP_MILLIS);
            } catch (InterruptedException e) {
                // ignore this and just loop
            }
        }

        // stop the thread pool for the processor
        tpProcessor.shutdown();
        
        // lose the reference to the thread pool processor
        tpProcessor = null;

        // lose the references to the input queue and output queue
        qInput = null;
        qOutput = null;
    }
}

/*
 * $Log$
 */
