/*  ProcessorThread.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

/**
 *  Defines a message processor thread.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class ProcessorThread implements Runnable {
    /**
     *  Idle thread sleep time (between waking up to check status).
     */
    protected int SLEEP_MILLIS = 1000;

    /**
     *  The processor thread.
     */
    protected Thread tProcessor;

    /**
     *  The message to process.
     */
    protected Message msg;

    /**
     *  The output queue which will receive message results (if any).
     */
    protected Queue qOutput;

    /**
     *  Thread pool that manages this processor thread.
     */
    protected ThreadPool tpManager;

    /**
     *  The last used time of this processor thread.
     */
    protected long lLastUsedMillis;

    /**
     *  The time when this will be idle too long and be shut down
     *  automatically.
     */
    protected long lShutdownMillis;

    /**
     *  Indicates if this is currently in use (active).
     */
    protected boolean bActive;

    /**
     *  Indicates if this is processing (thread started).
     */
    protected boolean bProcessing;

    /**
     *  The number of consecutive seconds this thread may be idle
     *  before being removed. If set to 0, this thread will never be
     *  removed because of idleness.
     */
    protected int iIdleLifetime;

    /**
     *  Create a new processor thread.
     */
    public ProcessorThread(ThreadPool tpManager, int iIdleLifetime) {
        this.tpManager = tpManager;
        this.iIdleLifetime = iIdleLifetime;
        tProcessor = new Thread(this);
        tProcessor.start();

        // set the last used time
        lLastUsedMillis = System.currentTimeMillis();
        
        // set the active flag
        bActive = false;
        
        // register this with the thread pool
        tpManager.registerProcessorThread(this);
    }

    /**
     *  Process a message.
     *
     *  @param msg the message to process.
     *  @param qOutput the output that will receive message results.
     */
    public void process(Message msg, Queue qOutput) {
        // flag this as active
        bActive = true;

        // set the current processor variables
        this.msg = msg;
        this.qOutput = qOutput;
    }
    
    /**
     *  Process the message and put the results (if any) on the output
     *  queue.
     */
    public void run() {
        // mark this as processing
        bProcessing = true;

        // check to see if this will shutdown if idle too long
        if (iIdleLifetime > 0) {
            while (bProcessing) {
                if (msg != null && qOutput != null) {
                    // TODO: flesh this out
                    qOutput.enqueue(msg);

                    // clear the processor variables
                    msg = null;
                    qOutput = null;
        
                    // set the last used time
                    lLastUsedMillis = System.currentTimeMillis();
                    lShutdownMillis = lLastUsedMillis + iIdleLifetime;

                    // set this as inactive
                    bActive = false;

                    // mark this as idle with the manager if this hasn't been
                    // shutdown
                    if (tProcessor != null) {
                        tpManager.nowFree(this);
                    }
                } 

                // check to see if this has been idle too long
                if (lShutdownMillis < System.currentTimeMillis()) {
                    // shut this down
                    shutdown();
                }
                
                // sleep for a period of time
                try {
                    tProcessor.sleep(SLEEP_MILLIS);
                } catch (InterruptedException e) {
                    // woken up, so we should have something to do
                }
            } // end while
        } else {
            // nonmanaged
            while (bProcessing) {
                if (msg != null && qOutput != null) {
                    // TODO: flesh this out
                    qOutput.enqueue(msg);

                    // clear the processor variables
                    msg = null;
                    qOutput = null;
        
                    // set the last used time
                    lLastUsedMillis = System.currentTimeMillis();

                    // set this as inactive
                    bActive = false;

                    // mark this as idle with the manager if this hasn't been
                    // shutdown
                    if (tProcessor != null) {
                        tpManager.nowFree(this);
                    }
                } 

                // sleep for a period of time
                try {
                    tProcessor.sleep(SLEEP_MILLIS);
                } catch (InterruptedException e) {
                    // woken up, so we should have something to do
                }
            } // end while
        }

        // unregister this (so it won't be reused)
        tpManager.unregisterProcessorThread(this);
    }

    /**
     *  Shut down the processor thread.
     */
    public void shutdown() {
        // stop this from processing
        bProcessing = false;

        // dereference the thread so it can be GCed
        tProcessor = null;
    }

    /**
     *  Indicates if this is actively processing.
     *
     *  @return true if this is active, false otherwise.
     */
    public boolean isActive() {
        return bActive;
    }

    /**
     *  Gets the last used time.
     *
     *  @return the last used time (in milliseconds).
     */
    public long getLastUsed() {
        return lLastUsedMillis;
    }
}
