/*  Queue.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 *  Implements a basic FIFO queue.
 *
 *  Uses a linked list to queue objects. Objects are
 *  retrieved from the head of the list, and enqueued at the end of
 *  the list.
 *
 *  Provides synchronization to make the queue thread-safe.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class Queue {
    /**
     *  The linked list containing the queue elements.
     */
    LinkedList list;

    /**
     *  Construct a new Queue.
     */
    public Queue() {
        super();
        list = new LinkedList();
    }

    /**
     *  Enqueue a new object.
     *
     *  @param o the object to be enqueued.
     */
    public void enqueue(Object o) {
        synchronized(list) {
            list.addLast(o);
        }
    }

    /**
     *  Dequeue the next object on the queue.
     *
     *  @return the next object available.
     *  @throws NoSuchElementException if the queue is empty.
     */
    public Object dequeue() throws NoSuchElementException {
        Object o;
        
        synchronized(list) {
            o = list.removeFirst();
        }

        return o;
    }

    /**
     *  Peeks at the next object available on the queue.
     *
     *  This method is not synchronized. If this queue is being 
     *  accessed by more than one thread, this
     *  object may be different from the one retrieved by dequeue().
     *
     *  @return the next object available.
     *  @throws NoSuchElementException if the queue is empty.
     */
    public Object peek() throws NoSuchElementException {
        return list.getFirst();
    }

    /**
     *  Get the number of elements in the queue.
     *
     *  This method is not synchronized and may return inconsistent
     *  results.
     *
     *  @return the number of elements in the queue.
     */
    public int getSize() {
        return list.size();
    }
}
