/*  ThreadPool.java
 *  Copyright (C) 2001 by Christopher R. Jones. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.mischiefbox.pollserve;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;

/**
 *  Implements a simple thread pool.
 *  <p>
 *  Tracks the in-use and available processor threads, holds onto idle
 *  processor threads, and creates new processor threads up to the
 *  maximum number of processor threads permitted. Releases idle
 *  processor threads after a specified period of time. Provides
 *  processor threads to callers.
 *  <p>
 *  Management is accomplished by a manager thread that examines the
 *  state of the thread pool occasionally.
 *
 *  @author Chris Jones
 *  @version $Id$
 */
public class ThreadPool implements Runnable {
    /**
     *  Time between thread pool checks. The number of milliseconds
     *  between each check of the thread pool.
     */
    protected static int SLEEP_MILLIS = 300;
    
    /**
     *  Idle thread ratio. The ratio of in-use threads to idle threads
     *  that should be maintained. Idle threads will not be collected
     *  until their minimum idle lifetime has been exceeded. New
     *  threads will be created when the ratio is too high.
     *  <p>
     *  To create as many threads as the maximum specified, set this
     *  to zero (which will disable thread management).
     *  <p>
     *  The default of 0.7 means that for 10 threads in the system,
     *  seven should be active at a time. If eight active threads are
     *  needed, a new idle thread will be created.
     */
    protected float fIdleRatio = 0.7F;

    /**
     *  The maximum idle thread lifetime. The number of concurrent
     *  milliseconds that a thread may be idle before being collected. If 
     *  set to 0, the idle thread will never be collected.
     */
    protected int iIdleLifetime = 30000;
    
    /**
     *  The maximum number of threads in the thread pool.
     */
    protected int iMaxThreads;
    
    /**
     *  The management thread.
     */
    protected Thread tManager;

    /**
     *  The available thread pool. Implemented as a queue (which will 
     *  permit round-robin access to threads in the pool).
     */
    protected Queue qThreads;

    /**
     *  The complete thread pool.
     */
    protected Vector vThreads;

    /**
     *  Indicates if this is managing the thread pool.
     */
    protected boolean bManaging;

    /**
     *  The current thread count (total).
     */
    protected int iThreadCount;

    /**
     *  The current number of active threads.
     */
    protected int iActiveCount;

    /**
     *  Create a new thread pool.
     *
     *  @param iMaxThreads the maximum number of threads in the thread
     *                     pool.
     */
    public ThreadPool(int iMaxThreads) {
        this.iMaxThreads = iMaxThreads;
        qThreads = new Queue();
        vThreads = new Vector(iMaxThreads);
        tManager = new Thread(this);
        tManager.start();
        iThreadCount = 0;
        iActiveCount = 0;
    }

    /**
     *  Create a new thread pool and specify the idle ratio.
     *
     *  @param iMaxThreads the maximum number of threads in the thread
     *                     pool.
     *  @param fIdleRatio the ratio of active threads to idle threads.
     */
    public ThreadPool(int iMaxThreads, float fIdleRatio) {
        this(iMaxThreads);
        this.fIdleRatio = fIdleRatio;
    }

    /**
     *  Create a new thread pool, specify the idle ratio, and indicate
     *  the maximum lifetime of an idle thread.
     *
     *  @param iMaxThreads the maximum number of threads in the thread
     *                     pool.
     *  @param fIdleRatio the ratio of active threads to idle threads.
     *  @param iIdleLifetime the maximum number of seconds a thread
     *                       may be in the idle state before being
     *                       killed.
     */
    public ThreadPool(int iMaxThreads, float fIdleRatio, 
                      int iIdleLifetime) {
        this(iMaxThreads, fIdleRatio);
        this.iIdleLifetime = iIdleLifetime;
    }
    
    /**
     *  Shut down the thread pool. Turn off thread management. Tell 
     *  each thread in the pool to shut down as soon as possible.
     *  <p>
     *  Synchronized.
     */
    public synchronized void shutdown() {
        // stop managing the threads
        bManaging = false;
        
        // get each thread and ask it to shutdown
        ProcessorThread pt;
        for (Enumeration en = vThreads.elements();
             en.hasMoreElements(); ) {
            pt = (ProcessorThread)en.nextElement();
            pt.shutdown();
        }
    }

    /**
     *  Register a newly created processor thread with the pool.
     *  <p>
     *  Synchronized access to the pool.
     *
     *  @param pt the new processor thread.
     */
    public void registerProcessorThread(ProcessorThread pt) {
        synchronized (vThreads) {
            vThreads.addElement(pt);
        }
        
        iThreadCount++;
    }

    /**
     *  Unregister a processor thread from the pool.
     *  <p>
     *  Synchronized access to the pool.
     *
     *  @param pt the processor thread to remove.
     */
    public void unregisterProcessorThread(ProcessorThread pt) {
        synchronized (vThreads) {
            vThreads.removeElement(pt);
        }
        
        iThreadCount--;
    }
    
    /**
     *  Get the next free thread from the pool. Blocks until the
     *  thread is available.
     *
     *  @return the next free processor thread.
     */
    public ProcessorThread nextFree() {
        // the processor thread to be returned
        ProcessorThread pt = null;
        
        while (pt == null) {
            try {
                // get the next available thread
                pt = (ProcessorThread)qThreads.dequeue();
                iActiveCount++;
            } catch (NoSuchElementException e) {
                // the thread wasn't available, so try again
                // TODO: consider sleeping here
            }
        }

        return pt;
    }

    /**
     *  Set a processor thread as free (idle).
     *  <p>
     *  Should only be called by the ProcessorThread.
     *
     *  @param pt the idle processor thread.
     */
    public void nowFree(ProcessorThread pt) {
        qThreads.enqueue(pt);
        iActiveCount--;
    }

    /**
     *  Get the next free thread from the pool. Non-blocking method.
     *
     *  @return the next free processor thread.
     *  @throws NoSuchElementException when there is no free processor
     *                                 thread.
     */
    public ProcessorThread getFree() throws NoSuchElementException {
        // the processor thread to be returned
        ProcessorThread pt = null;
        
        // get the next available thread (throw exception when queue
        // is empty)
        pt = (ProcessorThread)qThreads.dequeue();
        iActiveCount++;
        
        // this will only be returned if no exception was thrown
        return pt;
    }

    /**
     *  Manage the processor thread pool.
     *  <p>
     *  If we are managing threads, and if there are not enough idle 
     *  threads, and if the thread pool maximum size is not already 
     *  hit, create a new thread. If we are not managing threads, then
     *  make sure that there are enough threads in the pool, then end
     *  the loop.
     *  <p>
     *  If there are more idle threads than needed (and we care about
     *  the thread lifetime), then examine the idle threads and remove
     *  those that have exceeded their lifetimes.
     */
    public void run() {
        // the processor thread under examination
        ProcessorThread pt;

        // the minimum number of threads to keep
        int iMinThreads;
        
        // indicate that we are managing these threads
        bManaging = true;

        // calculate the minimum number of threads
        iMinThreads = (int)((1.0F - fIdleRatio) * 100) / iMaxThreads;

        if (iMinThreads < 0) {
            iMinThreads = 1;
        }

        // check to see if we are managing threads
        if (fIdleRatio != 0.0F) {
            // check the thread count to ensure enough are created
            while (iThreadCount < iMinThreads) {
                createProcessorThread();
            }

            // manage idle thread lifetimes
            while (bManaging) {
                // check the active ratio
                // idle threads are collected by the processor threads
                // themselves
                // check the active ratio
                if (((float)iActiveCount / (float)iThreadCount) > 
                    fIdleRatio &&
                    iActiveCount < iMaxThreads) {
                    // make a new thread to relieve the load
                    createProcessorThread();
                }

                if (iIdleLifetime < 1 &&
                    iActiveCount == iMaxThreads) {
                    // stop managing the thread pool since idle
                    // threads will never be collected
                    bManaging = false;
                }

                try {
                    tManager.sleep(SLEEP_MILLIS);
                } catch (InterruptedException e) {
                    // ignore this and just loop
                }
            } // end while
        } else {
            // don't manage threads...create up to MAX and leave it
            // there
            while (iThreadCount < iMaxThreads) {
                createProcessorThread();
            }
        }
    }

    /**
     *  Create a new processor thread.
     */
    protected void createProcessorThread() {
        // create the new processor thread--it registers itself if
        // successful
        ProcessorThread pt = new ProcessorThread(this, iIdleLifetime);

        // add the new processor thread to the idle queue
        qThreads.enqueue(pt);
    }
}

/*
 * $Log$
 */
